"""Code to use internally, for documentation."""
from __future__ import annotations

import io
from typing import Sequence, Union, get_args, get_origin

from docutils import nodes
from docutils.frontend import OptionParser
from docutils.parsers.rst import directives
from sphinx.directives import other
from sphinx.transforms.post_transforms import SphinxPostTransform
from sphinx.util import logging
from sphinx.util.docutils import SphinxDirective
#
from myst_parser.parsers.docutils_ import to_html5_demo
# from .config.main import MdParserConfig
# from .parsers.docutils_ import Parser as DocutilsParser
# from .warnings_ import MystWarnings

LOGGER = logging.getLogger(__name__)


class GutAdmonitionDirective(SphinxDirective):
    """Directive to show a set of admonitions, in a tab set."""

    required_arguments = 1
    final_argument_whitespace = True

    def run(self):
        """Run the directive."""
        types = [t.strip() for t in self.arguments[0].split(",")]
        content = "::::{tab-set}"
        for type_ in types:
            content += f"""
:::{{tab-item}} {type_}
````{{myst-example}}
```{{{type_}}}
Ceci est une admonition de type `{type_}`.
```
````
:::
"""
        content += "::::"
        node_ = nodes.Element()
        self.state.nested_parse(content.splitlines(), self.content_offset, node_)
        return node_.children


class GutSourcesDirective(SphinxDirective):
    """Directive for a Sources admonition.

    Memo: la transition (coupure thématique) doit être au plus haut
    niveau, et donc ne peut pas être incluse dans un container.

    https://docutils.sourceforge.io/docs/ref/rst/restructuredtext.html#transitions
    https://documatt.com/restructuredtext-reference/element/transition.html
    """

    has_content = True

    def run(self):
        """Run the directive."""
        content_str = "\n".join(self.content)
        content = ("___\n\n"
                   "::::{container} docusources\n\n"
                   ":::{admonition} Sources\n"
                   ":class: note\n"
                   f"\n{content_str}\n"
                   ":::\n"
                   "::::\n"
                   )
        node_ = nodes.Element()
        self.state.nested_parse(content.splitlines(), self.content_offset, node_)
        return node_.children


class GutLaTeXNoEditDirective(SphinxDirective):
    """Directive for non editable and not compilable LaTeX code blocks."""

    has_content = True

    def run(self):
        """Run the directive."""
        content_str = "\n".join(self.content)
        content = ":::{code-block}\n"
        content += ":class: noedit\n"
        content += f"""\
{content_str}
"""
        content += ":::"
        node_ = nodes.Element()
        self.state.nested_parse(content.splitlines(), self.content_offset, node_)
        return node_.children
