﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/chime-sdk-messaging/ChimeSDKMessagingRequest.h>
#include <aws/chime-sdk-messaging/ChimeSDKMessaging_EXPORTS.h>
#include <aws/chime-sdk-messaging/model/ChannelMessagePersistenceType.h>
#include <aws/chime-sdk-messaging/model/ChannelMessageType.h>
#include <aws/chime-sdk-messaging/model/MessageAttributeValue.h>
#include <aws/chime-sdk-messaging/model/PushNotificationConfiguration.h>
#include <aws/chime-sdk-messaging/model/Target.h>
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace ChimeSDKMessaging {
namespace Model {

/**
 */
class SendChannelMessageRequest : public ChimeSDKMessagingRequest {
 public:
  AWS_CHIMESDKMESSAGING_API SendChannelMessageRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "SendChannelMessage"; }

  AWS_CHIMESDKMESSAGING_API Aws::String SerializePayload() const override;

  AWS_CHIMESDKMESSAGING_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ARN of the channel.</p>
   */
  inline const Aws::String& GetChannelArn() const { return m_channelArn; }
  inline bool ChannelArnHasBeenSet() const { return m_channelArnHasBeenSet; }
  template <typename ChannelArnT = Aws::String>
  void SetChannelArn(ChannelArnT&& value) {
    m_channelArnHasBeenSet = true;
    m_channelArn = std::forward<ChannelArnT>(value);
  }
  template <typename ChannelArnT = Aws::String>
  SendChannelMessageRequest& WithChannelArn(ChannelArnT&& value) {
    SetChannelArn(std::forward<ChannelArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The content of the channel message.</p>
   */
  inline const Aws::String& GetContent() const { return m_content; }
  inline bool ContentHasBeenSet() const { return m_contentHasBeenSet; }
  template <typename ContentT = Aws::String>
  void SetContent(ContentT&& value) {
    m_contentHasBeenSet = true;
    m_content = std::forward<ContentT>(value);
  }
  template <typename ContentT = Aws::String>
  SendChannelMessageRequest& WithContent(ContentT&& value) {
    SetContent(std::forward<ContentT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of message, <code>STANDARD</code> or <code>CONTROL</code>.</p> <p>
   * <code>STANDARD</code> messages can be up to 4KB in size and contain metadata.
   * Metadata is arbitrary, and you can use it in a variety of ways, such as
   * containing a link to an attachment.</p> <p> <code>CONTROL</code> messages are
   * limited to 30 bytes and do not contain metadata.</p>
   */
  inline ChannelMessageType GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  inline void SetType(ChannelMessageType value) {
    m_typeHasBeenSet = true;
    m_type = value;
  }
  inline SendChannelMessageRequest& WithType(ChannelMessageType value) {
    SetType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Boolean that controls whether the message is persisted on the back end.
   * Required.</p>
   */
  inline ChannelMessagePersistenceType GetPersistence() const { return m_persistence; }
  inline bool PersistenceHasBeenSet() const { return m_persistenceHasBeenSet; }
  inline void SetPersistence(ChannelMessagePersistenceType value) {
    m_persistenceHasBeenSet = true;
    m_persistence = value;
  }
  inline SendChannelMessageRequest& WithPersistence(ChannelMessagePersistenceType value) {
    SetPersistence(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The optional metadata for each message.</p>
   */
  inline const Aws::String& GetMetadata() const { return m_metadata; }
  inline bool MetadataHasBeenSet() const { return m_metadataHasBeenSet; }
  template <typename MetadataT = Aws::String>
  void SetMetadata(MetadataT&& value) {
    m_metadataHasBeenSet = true;
    m_metadata = std::forward<MetadataT>(value);
  }
  template <typename MetadataT = Aws::String>
  SendChannelMessageRequest& WithMetadata(MetadataT&& value) {
    SetMetadata(std::forward<MetadataT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The <code>Idempotency</code> token for each client request.</p>
   */
  inline const Aws::String& GetClientRequestToken() const { return m_clientRequestToken; }
  inline bool ClientRequestTokenHasBeenSet() const { return m_clientRequestTokenHasBeenSet; }
  template <typename ClientRequestTokenT = Aws::String>
  void SetClientRequestToken(ClientRequestTokenT&& value) {
    m_clientRequestTokenHasBeenSet = true;
    m_clientRequestToken = std::forward<ClientRequestTokenT>(value);
  }
  template <typename ClientRequestTokenT = Aws::String>
  SendChannelMessageRequest& WithClientRequestToken(ClientRequestTokenT&& value) {
    SetClientRequestToken(std::forward<ClientRequestTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the <code>AppInstanceUser</code> or <code>AppInstanceBot</code>
   * that makes the API call.</p>
   */
  inline const Aws::String& GetChimeBearer() const { return m_chimeBearer; }
  inline bool ChimeBearerHasBeenSet() const { return m_chimeBearerHasBeenSet; }
  template <typename ChimeBearerT = Aws::String>
  void SetChimeBearer(ChimeBearerT&& value) {
    m_chimeBearerHasBeenSet = true;
    m_chimeBearer = std::forward<ChimeBearerT>(value);
  }
  template <typename ChimeBearerT = Aws::String>
  SendChannelMessageRequest& WithChimeBearer(ChimeBearerT&& value) {
    SetChimeBearer(std::forward<ChimeBearerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The push notification configuration of the message.</p>
   */
  inline const PushNotificationConfiguration& GetPushNotification() const { return m_pushNotification; }
  inline bool PushNotificationHasBeenSet() const { return m_pushNotificationHasBeenSet; }
  template <typename PushNotificationT = PushNotificationConfiguration>
  void SetPushNotification(PushNotificationT&& value) {
    m_pushNotificationHasBeenSet = true;
    m_pushNotification = std::forward<PushNotificationT>(value);
  }
  template <typename PushNotificationT = PushNotificationConfiguration>
  SendChannelMessageRequest& WithPushNotification(PushNotificationT&& value) {
    SetPushNotification(std::forward<PushNotificationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The attributes for the message, used for message filtering along with a
   * <code>FilterRule</code> defined in the
   * <code>PushNotificationPreferences</code>.</p>
   */
  inline const Aws::Map<Aws::String, MessageAttributeValue>& GetMessageAttributes() const { return m_messageAttributes; }
  inline bool MessageAttributesHasBeenSet() const { return m_messageAttributesHasBeenSet; }
  template <typename MessageAttributesT = Aws::Map<Aws::String, MessageAttributeValue>>
  void SetMessageAttributes(MessageAttributesT&& value) {
    m_messageAttributesHasBeenSet = true;
    m_messageAttributes = std::forward<MessageAttributesT>(value);
  }
  template <typename MessageAttributesT = Aws::Map<Aws::String, MessageAttributeValue>>
  SendChannelMessageRequest& WithMessageAttributes(MessageAttributesT&& value) {
    SetMessageAttributes(std::forward<MessageAttributesT>(value));
    return *this;
  }
  template <typename MessageAttributesKeyT = Aws::String, typename MessageAttributesValueT = MessageAttributeValue>
  SendChannelMessageRequest& AddMessageAttributes(MessageAttributesKeyT&& key, MessageAttributesValueT&& value) {
    m_messageAttributesHasBeenSet = true;
    m_messageAttributes.emplace(std::forward<MessageAttributesKeyT>(key), std::forward<MessageAttributesValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the SubChannel in the request.</p>
   */
  inline const Aws::String& GetSubChannelId() const { return m_subChannelId; }
  inline bool SubChannelIdHasBeenSet() const { return m_subChannelIdHasBeenSet; }
  template <typename SubChannelIdT = Aws::String>
  void SetSubChannelId(SubChannelIdT&& value) {
    m_subChannelIdHasBeenSet = true;
    m_subChannelId = std::forward<SubChannelIdT>(value);
  }
  template <typename SubChannelIdT = Aws::String>
  SendChannelMessageRequest& WithSubChannelId(SubChannelIdT&& value) {
    SetSubChannelId(std::forward<SubChannelIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The content type of the channel message.</p>
   */
  inline const Aws::String& GetContentType() const { return m_contentType; }
  inline bool ContentTypeHasBeenSet() const { return m_contentTypeHasBeenSet; }
  template <typename ContentTypeT = Aws::String>
  void SetContentType(ContentTypeT&& value) {
    m_contentTypeHasBeenSet = true;
    m_contentType = std::forward<ContentTypeT>(value);
  }
  template <typename ContentTypeT = Aws::String>
  SendChannelMessageRequest& WithContentType(ContentTypeT&& value) {
    SetContentType(std::forward<ContentTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The target of a message. Must be a member of the channel, such as another
   * user, a bot, or the sender. Only the target and the sender can view targeted
   * messages. Only users who can see targeted messages can take actions on them.
   * However, administrators can delete targeted messages that they can’t see. </p>
   */
  inline const Aws::Vector<Target>& GetTarget() const { return m_target; }
  inline bool TargetHasBeenSet() const { return m_targetHasBeenSet; }
  template <typename TargetT = Aws::Vector<Target>>
  void SetTarget(TargetT&& value) {
    m_targetHasBeenSet = true;
    m_target = std::forward<TargetT>(value);
  }
  template <typename TargetT = Aws::Vector<Target>>
  SendChannelMessageRequest& WithTarget(TargetT&& value) {
    SetTarget(std::forward<TargetT>(value));
    return *this;
  }
  template <typename TargetT = Target>
  SendChannelMessageRequest& AddTarget(TargetT&& value) {
    m_targetHasBeenSet = true;
    m_target.emplace_back(std::forward<TargetT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_channelArn;
  bool m_channelArnHasBeenSet = false;

  Aws::String m_content;
  bool m_contentHasBeenSet = false;

  ChannelMessageType m_type{ChannelMessageType::NOT_SET};
  bool m_typeHasBeenSet = false;

  ChannelMessagePersistenceType m_persistence{ChannelMessagePersistenceType::NOT_SET};
  bool m_persistenceHasBeenSet = false;

  Aws::String m_metadata;
  bool m_metadataHasBeenSet = false;

  Aws::String m_clientRequestToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientRequestTokenHasBeenSet = true;

  Aws::String m_chimeBearer;
  bool m_chimeBearerHasBeenSet = false;

  PushNotificationConfiguration m_pushNotification;
  bool m_pushNotificationHasBeenSet = false;

  Aws::Map<Aws::String, MessageAttributeValue> m_messageAttributes;
  bool m_messageAttributesHasBeenSet = false;

  Aws::String m_subChannelId;
  bool m_subChannelIdHasBeenSet = false;

  Aws::String m_contentType;
  bool m_contentTypeHasBeenSet = false;

  Aws::Vector<Target> m_target;
  bool m_targetHasBeenSet = false;
};

}  // namespace Model
}  // namespace ChimeSDKMessaging
}  // namespace Aws
