/*******************************************************************************
* Copyright (C) 2019 Intel Corporation
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*
*  Content:
*       This example demonstrates use of oneAPI Math Kernel Library (oneMKL)
*       DPCPP USM API oneapi::mkl::sparse::gemv to perform general
*       sparse matrix-vector  multiplication on a SYCL device (CPU, GPU). This
*       example uses a sparse matrix in CSR format.
*
*       y = alpha * op(A) * x + beta * y
*
*       where op() is defined by one of
*           oneapi::mkl::transpose::{nontrans,trans,conjtrans}
*
*       The supported floating point data types for gemv are:
*           float
*           double
*           std::complex<float>
*           std::complex<double>
*
*       The supported matrix formats for gemv are:
*           CSR
*           CSC
*           COO
*           BSR
*
*******************************************************************************/

// stl includes
#include <algorithm>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <iterator>
#include <limits>
#include <list>
#include <vector>

#include "mkl.h"
#include "oneapi/mkl.hpp"
#include <sycl/sycl.hpp>

// local includes
#include "common_for_examples.hpp"
#include "./include/common_for_sparse_examples.hpp"

//
// Main example for Sparse Matrix-Vector Multiply consisting of
// initialization of A matrix, x and y vectors as well as
// scalars alpha and beta.  Then the product
//
// y = alpha * op(A) * x + beta * y
//
// is performed and finally the results are post processed.
//
template <typename dataType, typename intType>
int run_sparse_blas_example(sycl::queue &q)
{
    bool good = true;

    // handle for sparse matrix
    oneapi::mkl::sparse::matrix_handle_t csrA = nullptr;

    // create arrays to handle deallocation
    std::vector<intType *> int_ptr_vec;
    std::vector<dataType *> data_ptr_vec;

    try {
        // Initialize data for Sparse Matrix-Vector Multiply
        auto opA = oneapi::mkl::transpose::trans;
        auto index_base_val = oneapi::mkl::index_base::zero;
        if constexpr (is_complex<dataType>()) {
            opA = oneapi::mkl::transpose::conjtrans;
            index_base_val = oneapi::mkl::index_base::one;
        }
        intType a_ind = index_base_val == oneapi::mkl::index_base::zero ? 0 : 1;

        // Matrix data size
        intType size  = 4;
        const std::int64_t nrows = size * size * size;
        const std::int64_t ncols = nrows;

        // Set scalar dataType values
        dataType alpha, beta;
        if constexpr (is_complex<dataType>()) {
            alpha = set_fp_value(dataType(1.0), dataType(0.0));
            beta  = set_fp_value(dataType(0.0), dataType(1.0));
        }
        else {
            alpha = set_fp_value(dataType(1.0), dataType(0.0));
            beta  = set_fp_value(dataType(0.0), dataType(0.0));
        }

        intType *ia, *ja;
        dataType *a, *x, *y, *y_ref;
        const std::int64_t sizea   = 27 * nrows;
        const std::int64_t sizeja  = 27 * nrows;
        const std::int64_t sizeia  = nrows + 1;
        const std::int64_t sizevec = nrows;

        ia    = sycl::malloc_shared<intType>(sizeia, q);
        ja    = sycl::malloc_shared<intType>(sizeja, q);
        a     = sycl::malloc_shared<dataType>(sizea, q);
        x     = sycl::malloc_shared<dataType>(sizevec, q);
        y     = sycl::malloc_shared<dataType>(sizevec, q);
        y_ref = sycl::malloc_shared<dataType>(sizevec, q);

        if (!ia || !ja || !a || !x || !y || !y_ref) {
            std::string errorMessage =
                "Failed to allocate USM shared memory arrays \n"
                " for CSR A matrix: ia(" + std::to_string((sizeia)*sizeof(intType)) + " bytes)\n"
                "                   ja(" + std::to_string((sizeja)*sizeof(intType)) + " bytes)\n"
                "                   a(" + std::to_string((sizea)*sizeof(dataType)) + " bytes)\n"
                " and vectors:      x(" + std::to_string((sizevec)*sizeof(dataType)) + " bytes)\n"
                "                   y(" + std::to_string((sizevec)*sizeof(dataType)) + " bytes)\n"
                "                   y_ref(" + std::to_string((sizevec)*sizeof(dataType)) + " bytes)";

            throw std::runtime_error(errorMessage);
        }
        int_ptr_vec.push_back(ia);
        int_ptr_vec.push_back(ja);
        data_ptr_vec.push_back(a);
        data_ptr_vec.push_back(x);
        data_ptr_vec.push_back(y);
        data_ptr_vec.push_back(y_ref);

        generate_sparse_matrix<dataType, intType>(size, ia, ja, a, a_ind);
        const std::int64_t nnz = ia[nrows] - a_ind;

        // Init vectors x and y
        for (int i = 0; i < nrows; i++) {
            x[i] = set_fp_value(dataType(1.0), dataType(0.0));
            y[i] = set_fp_value(dataType(0.0), dataType(0.0));
            y_ref[i] = set_fp_value(dataType(0.0), dataType(0.0));
        }

        //
        // Execute Sparse Matrix - Dense Vector Multiply
        //

        std::cout << "\n\t\tsparse::gemv parameters:\n";
        std::cout << "\t\t\ttransA = "
                  << (opA == oneapi::mkl::transpose::nontrans ?
                              "nontrans" :
                              (opA == oneapi::mkl::transpose::trans ? "trans" : "conjtrans"))
                  << std::endl;
        std::cout << "\t\t\ta_nrows = " << nrows << std::endl;
        std::cout << "\t\t\ta_ncols = " << ncols << std::endl;
        std::cout << "\t\t\ta_nnz   = " << nnz << std::endl;
        std::cout << "\t\t\talpha   = " << alpha << std::endl;
        std::cout << "\t\t\tbeta    = " << beta << std::endl;


        // create and initialize handle for a Sparse Matrix in CSR format

        oneapi::mkl::sparse::init_matrix_handle(&csrA);

        auto ev_set = oneapi::mkl::sparse::set_csr_data(q, csrA, nrows, ncols, nnz, index_base_val, ia, ja, a);

        auto ev_optmv = oneapi::mkl::sparse::optimize_gemv(q, opA, csrA, {ev_set});

        auto ev_gemv = oneapi::mkl::sparse::gemv(q, opA, alpha, csrA, x, beta, y, {ev_optmv});

        auto ev_release = oneapi::mkl::sparse::release_matrix_handle(q, &csrA, {ev_gemv});

        ev_release.wait();

        //
        // Post Processing
        //

        dataType *res = y;
        const bool isConj = (opA == oneapi::mkl::transpose::conjtrans);
        for (intType row = 0; row < nrows; row++) {
            y_ref[row] *= beta;
        }
        if (opA == oneapi::mkl::transpose::nontrans) {
            for (intType row = 0; row < nrows; row++) {
                dataType tmp = dataType(0.0);
                for (intType i = ia[row] - a_ind; i < ia[row + 1]-a_ind; i++) {
                    const intType col = ja[i] - a_ind;
                    tmp += a[i] * x[col];
                }
                y_ref[row] += alpha * tmp;
            }
        }
        else { // opA is trans or conjtrans
            for (intType row = 0; row < nrows; row++) {
                dataType tmp = alpha * x[row];
                for (intType i = ia[row]-a_ind; i < ia[row + 1]-a_ind; i++) {
                    if constexpr (is_complex<dataType>()) {
                        y_ref[ja[i]-a_ind] += tmp * (isConj ? std::conj(a[i]) : a[i]);
                    }
                    else {
                        y_ref[ja[i]-a_ind] += tmp * a[i];
                    }
                }
            }
        }

        for (intType row = 0; row < nrows; row++) {
            good &= check_result(res[row], y_ref[row], nrows, row);
        }

        std::cout << "\n\t\t sparse::gemv example " << (good ? "passed" : "failed")
                  << "\n\tFinished" << std::endl;

        q.wait_and_throw();

    }
    catch (sycl::exception const &e) {
        std::cout << "\t\tCaught synchronous SYCL exception:\n" << e.what() << std::endl;
        good = false;
    }
    catch (std::exception const &e) {
        std::cout << "\t\tCaught std exception:\n" << e.what() << std::endl;
        good = false;
    }

    q.wait();

    // backup cleaning of release_handle if exceptions happened
    oneapi::mkl::sparse::release_matrix_handle(q, &csrA).wait();

    // cleanup allocations
    cleanup_arrays<dataType, intType>(data_ptr_vec, int_ptr_vec, q);

    q.wait();

    return good ? 0 : 1;
}

//
// Description of example setup, apis used and supported floating point type
// precisions
//
void print_example_banner()
{

    std::cout << "" << std::endl;
    std::cout << "###############################################################"
                 "#########"
              << std::endl;
    std::cout << "# Sparse Matrix-Vector Multiply Example: " << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# y = alpha * op(A) * x + beta * y" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# where A is a sparse matrix in CSR format, x and y are "
                 "dense vectors"
              << std::endl;
    std::cout << "# and alpha, beta are floating point type precision scalars." << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Using apis:" << std::endl;
    std::cout << "#   sparse::gemv" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Supported floating point type precisions:" << std::endl;
    std::cout << "#   float" << std::endl;
    std::cout << "#   double" << std::endl;
    std::cout << "#   std::complex<float>" << std::endl;
    std::cout << "#   std::complex<double>" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "###############################################################"
                 "#########"
              << std::endl;
    std::cout << std::endl;
}

//
// Main entry point for example.
//
// Dispatches to appropriate device types as set at build time with flag:
// -DSYCL_DEVICES_cpu -- only runs SYCL CPU implementation
// -DSYCL_DEVICES_gpu -- only runs SYCL GPU implementation
// -DSYCL_DEVICES_all (default) -- runs on all: cpu and gpu devices
//
//  For each device selected and each supported data type,
//  run_sparse_blas_example is run with all supported data types,
//  if any fail, we move on to the next device.
//

int main(int argc, char **argv)
{

    print_example_banner();

    std::list<my_sycl_device_types> list_of_devices;
    set_list_of_devices(list_of_devices);

    int status = 0;
    for (auto it = list_of_devices.begin(); it != list_of_devices.end(); ++it) {
        try {
            sycl::device my_dev;
            bool my_dev_is_found = false;
            get_sycl_device(my_dev, my_dev_is_found, *it);

            if (my_dev_is_found) {
                std::cout << "Running tests on " << sycl_device_names[*it] << ".\n";

                // Catch asynchronous exceptions
                auto exception_handler = [](sycl::exception_list exceptions) {
                    for (std::exception_ptr const &e : exceptions) {
                        try {
                            std::rethrow_exception(e);
                        }
                        catch (sycl::exception const &e) {
                            std::cout << "Caught asynchronous SYCL exception: \n"
                                << e.what() << std::endl;
                        }
                    }
                };

                sycl::queue q(my_dev, exception_handler);

                std::cout << "\tRunning with single precision real data type:" << std::endl;
                status |= run_sparse_blas_example<float, std::int32_t>(q);

                if (my_dev.get_info<sycl::info::device::double_fp_config>().size() != 0) {
                    std::cout << "\tRunning with double precision real data type:" << std::endl;
                    status |= run_sparse_blas_example<double, std::int32_t>(q);
                }

                std::cout << "\tRunning with single precision complex data type:" << std::endl;
                status |= run_sparse_blas_example<std::complex<float>, std::int32_t>(q);

                if (my_dev.get_info<sycl::info::device::double_fp_config>().size() != 0) {
                    std::cout << "\tRunning with double precision complex data type:" << std::endl;
                    status |= run_sparse_blas_example<std::complex<double>, std::int32_t>(q);
                }

            }
            else {
#ifdef FAIL_ON_MISSING_DEVICES
                std::cout << "No " << sycl_device_names[*it]
                    << " devices found; Fail on missing devices "
                    "is enabled.\n";
                return 1;
#else
                std::cout << "No " << sycl_device_names[*it] << " devices found; skipping "
                    << sycl_device_names[*it] << " tests.\n";
#endif
            }
        }
        catch (sycl::exception const &e) {
            std::cout << "\t\tCaught SYCL exception at driver level: \n" << e.what() << std::endl;
            continue; // stop with device, but move on to other devices
        }
        catch (std::exception const &e) {
            std::cout << "\t\tCaught std exception at driver level: \n" << e.what() << std::endl;
            continue; // stop with device, but move on to other devices
        }


    } // for loop over devices

    mkl_free_buffers();
    return status;
}
