/*******************************************************************************
* Copyright (C) 2018 Intel Corporation
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
 *
 *  Content:
 *            Template for VM issues reproducer
 *            performing one function call at defined arguments
 *            and simple results comparison to reference values.
 *
 *******************************************************************************/

#include <cmath>
#include <cstdio>
#include <stdexcept>

#include "oneapi/mkl.hpp"
#include <sycl/sycl.hpp>

#include "common_for_examples.hpp"

// VM function
#define FUNC erf
// Data type
#define TYPE double
// Accuracy
#define ACC la
// Vector length
#define LEN 2
// Input arguments
#define ARGS 0.70710678118654752, 1.41421356237309504
// Expected reference results
#define REFS 0.68268949213708596, 0.95449973610364158
// Allowed relative error
#define EPS 1.0e-15

// String making macro for print
#define str(x) #x
#define STR(x) str(x)

/**
 * @brief Testing preamble.
 *
 * Print device aand driver info.
 *
 * @param[in] dev    Sycl device
 *
 */
void own_preamble(sycl::device& dev) {
    std::string dev_name = dev.template get_info<sycl::info::device::name>();
    std::string driver_version = dev.template get_info<sycl::info::device::version>();
    fprintf(stdout, "\t       device name: %s\n", dev_name.c_str());
    fprintf(stdout, "\t    driver version: %s\n\n", driver_version.c_str());
    fflush(stdout);
    return;
}

/**
 * @brief Asynchronous error handler.
 *
 * Async sycl error catching procedure.
 *
 * @param[in] el Exceptions list
 *
 */
void own_async_sycl_error(sycl::exception_list el) {
    fprintf(stderr, "async exceptions caught: \n");

    for (auto l = el.begin(); l != el.end(); l = l + 1) {
        try {
            std::rethrow_exception(*l);
        } catch (const sycl::exception& e) {
            fprintf(stderr, "SYCL exception occured with code %d with %s\n", e.code().value(), e.what());
        }
    }
    return;
} // own_async_sycl_error

/**
 * @brief Template for VM reproducer
 *
 * Performs call one VM function and simple results check
 *
 * @param[in] dev    Sycl device
 *
 * @return           Status (-1 for errors)
 *
 */
int own_run_on(sycl::device& dev) {
    int err = 0;            // Number of errors
    TYPE arg[LEN] = {ARGS}; // Input array
    TYPE ref[LEN] = {REFS}; // Reference results array
    TYPE res[LEN] = {0};    // Output array

    own_preamble(dev);

    // Sycl device queue
    sycl::queue queue{dev, own_async_sycl_error};

    // VM call
    auto ev1 = oneapi::mkl::vm::FUNC(queue, LEN, arg, res, {}, oneapi::mkl::vm::mode::ACC);
    // Wait for asunc execution complete
    ev1.wait();

    for (int i = 0; i < LEN; i = i + 1) {
        fprintf(stderr, "\t%s (%.17lg) = \n\t\t%.17lg / computed /\n\t\t%.17lg / expected /\n", STR(FUNC), arg[i], res[i], ref[i]);

        // Check relative error versus reference result
        err += (fabs((res[i] - ref[i]) / ref[i]) > EPS);
    }

    return (err) ? -1 : 0;
} // own_run_on

//
// Main entry point for example.
//
// Dispatches to appropriate device types as set at build time with flag:
// -DSYCL_DEVICES_cpu -- only runs SYCL CPU device
// -DSYCL_DEVICES_gpu -- only runs SYCL GPU device
// -DSYCL_DEVICES_all (default) -- runs on all: CPU and GPU devices
//
//  For each device selected and each data type supported, the example
//  runs with all supported data types
//
int main(int argc, char** argv) {
    int ret = 0; // return status
    fprintf(stdout, "sycl vm_host_one_func: started...\n");
    fflush(stdout);

    // List of available devices
    std::list<my_sycl_device_types> list_of_devices;
    set_list_of_devices(list_of_devices);

    // Loop by all available devices
    for (auto dev_type : list_of_devices) {
        sycl::device my_dev;
        bool my_dev_is_found = false;
        get_sycl_device(my_dev, my_dev_is_found, dev_type);

        // Run tests if the device is available
        if (my_dev_is_found) {
            fprintf(stdout, "Running tests on %s.\n", sycl_device_names[dev_type].c_str());
            fflush(stdout);
            try {
                ret |= own_run_on(my_dev);
            } catch (sycl::exception const& e) {
                fprintf(stderr, "sycl::exception caught. %s\n", e.what());
                ret = 1;
            } catch (std::exception const& e) {
                fprintf(stderr, "std::exception caught. %s\n", e.what());
                ret = 1;
            }
        } else {
            fprintf(stderr, "No %s devices found; skipping %s tests.\n", sycl_device_names[dev_type].c_str(),
                    sycl_device_names[dev_type].c_str());
        }
    }

    fflush(stdout);
    fprintf(stdout, "sycl vm_host_one_func: %s\n\n", (ret != 0) ? "FAIL" : "PASS");
    return ret;
}
