#
# Copyright (C) 2017 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of them
# is governed by the express license under which they were provided to you ("License"). Unless
# the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
# or transmit this software or the related documents without Intel's prior written permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#


# ------------------------------------------------------------------------------
# This example shows how to obtain detailed dependency and MAP data from the
# bottomup table.
# ------------------------------------------------------------------------------

import sys

try:

    import advisor

except ImportError:

    print(
        """Import error: Python could not resolve path to Advisor's pythonapi directory.
        To fix, either manually add path to the pythonapi directory into PYTHONPATH environment
        variable, or use advixe-vars.* scripts to set up product environment variables automatically."""
    )
    sys.exit(1)

# Check command-line arguments.
if len(sys.argv) < 2:
    print('Usage: "python {} path_to_project_dir"'.format(__file__))
    sys.exit(2)

# Open the Advisor Project.
project = advisor.open_project(sys.argv[1])
# Load vectorization data (advisor.VECTORIZATION_DATA is equivalent to advisor.SURVEY | advisor.DEPENDENCIES | advisor.MAP).
# advisor.VECTORIZATION_DATA also loads advisor.RECOMMENDATIONS. We exclude recommendations in order to speed up loading.
data = project.load(advisor.VECTORIZATION_DATA ^ advisor.RECOMMENDATIONS)

# Indentation for pretty-printing.
indent = "   "

# Iteratve over each entry in the bottomup table.
for loop in data.bottomup:
    print(loop["mangled_name"])

    # Each loop may have several dependency sites, each site has problems, and
    # each problem has observations.
    for site in loop.dependencies:
        print(indent + "dependency site: {}, {}".format(site["site_name"], site["source"]))
        for problem in site.problems:
            print(indent * 2 + "problem: {}, {}".format(problem["id"], problem["type"]))
            for observation in problem.observations:
                print(indent * 3 + "observation: {}, {}".format(observation["id"], observation["description"]))

    # Each loop may have several MAP sites, each site have several problems and
    # also each site may have cache simulation info.
    for site in loop.map:
        print(indent + "map site: {}, {}".format(site["site_name"], site["source"]))
        for problem in site.problems:
            print(
                "{}problem: {}, {}, {}".format(
                    indent * 2, problem["id"], problem["access_pattern_type"], problem["source"]
                )
            )
