﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/sesv2/SESV2_EXPORTS.h>
#include <aws/sesv2/model/RecommendationImpact.h>
#include <aws/sesv2/model/RecommendationStatus.h>
#include <aws/sesv2/model/RecommendationType.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace SESV2 {
namespace Model {

/**
 * <p>A recommendation generated for your account.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/Recommendation">AWS
 * API Reference</a></p>
 */
class Recommendation {
 public:
  AWS_SESV2_API Recommendation() = default;
  AWS_SESV2_API Recommendation(Aws::Utils::Json::JsonView jsonValue);
  AWS_SESV2_API Recommendation& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_SESV2_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The resource affected by the recommendation, with values like
   * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>.</p>
   */
  inline const Aws::String& GetResourceArn() const { return m_resourceArn; }
  inline bool ResourceArnHasBeenSet() const { return m_resourceArnHasBeenSet; }
  template <typename ResourceArnT = Aws::String>
  void SetResourceArn(ResourceArnT&& value) {
    m_resourceArnHasBeenSet = true;
    m_resourceArn = std::forward<ResourceArnT>(value);
  }
  template <typename ResourceArnT = Aws::String>
  Recommendation& WithResourceArn(ResourceArnT&& value) {
    SetResourceArn(std::forward<ResourceArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The recommendation type, with values like <code>DKIM</code>,
   * <code>SPF</code>, <code>DMARC</code>, <code>BIMI</code>, or
   * <code>COMPLAINT</code>.</p>
   */
  inline RecommendationType GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  inline void SetType(RecommendationType value) {
    m_typeHasBeenSet = true;
    m_type = value;
  }
  inline Recommendation& WithType(RecommendationType value) {
    SetType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The recommendation description / disambiguator - e.g. <code>DKIM1</code> and
   * <code>DKIM2</code> are different recommendations about your DKIM setup.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  Recommendation& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The recommendation status, with values like <code>OPEN</code> or
   * <code>FIXED</code>.</p>
   */
  inline RecommendationStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(RecommendationStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline Recommendation& WithStatus(RecommendationStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The first time this issue was encountered and the recommendation was
   * generated.</p>
   */
  inline const Aws::Utils::DateTime& GetCreatedTimestamp() const { return m_createdTimestamp; }
  inline bool CreatedTimestampHasBeenSet() const { return m_createdTimestampHasBeenSet; }
  template <typename CreatedTimestampT = Aws::Utils::DateTime>
  void SetCreatedTimestamp(CreatedTimestampT&& value) {
    m_createdTimestampHasBeenSet = true;
    m_createdTimestamp = std::forward<CreatedTimestampT>(value);
  }
  template <typename CreatedTimestampT = Aws::Utils::DateTime>
  Recommendation& WithCreatedTimestamp(CreatedTimestampT&& value) {
    SetCreatedTimestamp(std::forward<CreatedTimestampT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The last time the recommendation was updated.</p>
   */
  inline const Aws::Utils::DateTime& GetLastUpdatedTimestamp() const { return m_lastUpdatedTimestamp; }
  inline bool LastUpdatedTimestampHasBeenSet() const { return m_lastUpdatedTimestampHasBeenSet; }
  template <typename LastUpdatedTimestampT = Aws::Utils::DateTime>
  void SetLastUpdatedTimestamp(LastUpdatedTimestampT&& value) {
    m_lastUpdatedTimestampHasBeenSet = true;
    m_lastUpdatedTimestamp = std::forward<LastUpdatedTimestampT>(value);
  }
  template <typename LastUpdatedTimestampT = Aws::Utils::DateTime>
  Recommendation& WithLastUpdatedTimestamp(LastUpdatedTimestampT&& value) {
    SetLastUpdatedTimestamp(std::forward<LastUpdatedTimestampT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The recommendation impact, with values like <code>HIGH</code> or
   * <code>LOW</code>.</p>
   */
  inline RecommendationImpact GetImpact() const { return m_impact; }
  inline bool ImpactHasBeenSet() const { return m_impactHasBeenSet; }
  inline void SetImpact(RecommendationImpact value) {
    m_impactHasBeenSet = true;
    m_impact = value;
  }
  inline Recommendation& WithImpact(RecommendationImpact value) {
    SetImpact(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_resourceArn;

  RecommendationType m_type{RecommendationType::NOT_SET};

  Aws::String m_description;

  RecommendationStatus m_status{RecommendationStatus::NOT_SET};

  Aws::Utils::DateTime m_createdTimestamp{};

  Aws::Utils::DateTime m_lastUpdatedTimestamp{};

  RecommendationImpact m_impact{RecommendationImpact::NOT_SET};
  bool m_resourceArnHasBeenSet = false;
  bool m_typeHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_createdTimestampHasBeenSet = false;
  bool m_lastUpdatedTimestampHasBeenSet = false;
  bool m_impactHasBeenSet = false;
};

}  // namespace Model
}  // namespace SESV2
}  // namespace Aws
