import { GrafanaManagedContactPoint } from 'app/plugins/datasource/alertmanager/types';

import { KnownProvenance } from '../../types/knownProvenance';
import { ReceiverTypes } from '../receivers/grafanaAppReceivers/onCall/onCall';

import { RECEIVER_META_KEY, RECEIVER_PLUGIN_META_KEY } from './constants';
import {
  ReceiverConfigWithMetadata,
  enhanceContactPointsWithMetadata,
  getReceiverDescription,
  isAutoGeneratedPolicy,
  summarizeEmailAddresses,
} from './utils';

describe('isAutoGeneratedPolicy', () => {
  it('should return false when not enabled', () => {
    expect(isAutoGeneratedPolicy({})).toBe(false);
  });
});

describe('getReceiverDescription', () => {
  it('should show multiple email addresses', () => {
    const receiver: ReceiverConfigWithMetadata = {
      name: 'email',
      provenance: undefined,
      type: 'email',
      disableResolveMessage: false,
      settings: { addresses: 'test1@test.com,test2@test.com,test3@test.com,test4@test.com' },
      [RECEIVER_META_KEY]: {
        name: 'Email',
        description: 'The email receiver',
      },
    };

    expect(getReceiverDescription(receiver)).toBe('test1@test.com, test2@test.com, test3@test.com, +1 more');
  });

  it('should work for Slack', () => {
    const output = '#channel';
    const receiver1: ReceiverConfigWithMetadata = {
      name: 'slack',
      provenance: undefined,
      type: 'slack',
      disableResolveMessage: false,
      settings: { recipient: '#channel' },
      [RECEIVER_META_KEY]: {
        name: 'Slack',
        description: 'The Slack receiver',
      },
    };

    const receiver2: ReceiverConfigWithMetadata = {
      name: 'slack',
      provenance: undefined,
      type: 'slack',
      disableResolveMessage: false,
      settings: { recipient: 'channel' },
      [RECEIVER_META_KEY]: {
        name: 'Slack',
        description: 'The Slack receiver',
      },
    };

    expect(getReceiverDescription(receiver1)).toBe(output);
    expect(getReceiverDescription(receiver2)).toBe(output);
  });

  it('should work for OnCall', () => {
    const output = 'The OnCall receiver';
    const input: ReceiverConfigWithMetadata = {
      name: 'my oncall',
      provenance: undefined,
      type: ReceiverTypes.OnCall,
      disableResolveMessage: false,
      settings: {},
      [RECEIVER_PLUGIN_META_KEY]: {
        description: output,
        icon: '',
        title: '',
      },
      [RECEIVER_META_KEY]: {
        name: '',
      },
    };

    expect(getReceiverDescription(input)).toBe(output);
  });

  it('should work for any type', () => {
    const output = 'Some description of the receiver';
    const input: ReceiverConfigWithMetadata = {
      name: 'some receiver',
      provenance: undefined,
      type: 'some',
      disableResolveMessage: false,
      settings: {},
      [RECEIVER_META_KEY]: {
        name: 'Some Receiver',
        description: output,
      },
    };

    expect(getReceiverDescription(input)).toBe(output);
  });

  it('should work for any type with no description', () => {
    const input: ReceiverConfigWithMetadata = {
      name: 'some receiver',
      provenance: undefined,
      type: 'some',
      disableResolveMessage: false,
      settings: {},
      [RECEIVER_META_KEY]: {
        name: 'Some Receiver',
      },
    };

    expect(getReceiverDescription(input)).toBe(undefined);
  });
});

describe('summarizeEmailAddresses', () => {
  it('should work with one email address', () => {
    expect(summarizeEmailAddresses('test@test.com')).toBe('test@test.com');
  });

  it('should work with multiple types of separators', () => {
    const output = 'foo@foo.com, bar@bar.com';

    expect(summarizeEmailAddresses('foo@foo.com,   bar@bar.com')).toBe(output);
    expect(summarizeEmailAddresses(' foo@foo.com;  bar@bar.com')).toBe(output);
    expect(summarizeEmailAddresses('foo@foo.com\n bar@bar.com  ')).toBe(output);
  });
});

describe('enhanceContactPointsWithMetadata', () => {
  it('should extract provenance from receiver configs when contact point has no provenance', () => {
    const contactPoint: GrafanaManagedContactPoint = {
      name: 'test-contact-point',
      grafana_managed_receiver_configs: [
        {
          uid: 'test-uid',
          name: 'test-contact-point',
          type: 'email',
          settings: { addresses: 'test@example.com' },
          secureFields: {},
          provenance: KnownProvenance.API,
        },
      ],
    };

    const enhanced = enhanceContactPointsWithMetadata({
      contactPoints: [contactPoint],
      notifiers: [],
      status: [],
    });

    expect(enhanced[0].provenance).toBe(KnownProvenance.API);
  });

  it('should prefer contact point provenance over receiver config provenance', () => {
    const contactPoint: GrafanaManagedContactPoint = {
      name: 'test-contact-point',
      provenance: KnownProvenance.File, // Provenance on contact point (from K8s)
      grafana_managed_receiver_configs: [
        {
          uid: 'test-uid',
          name: 'test-contact-point',
          type: 'email',
          settings: { addresses: 'test@example.com' },
          secureFields: {},
          provenance: KnownProvenance.API, // Different provenance on receiver config
        },
      ],
    };

    const enhanced = enhanceContactPointsWithMetadata({
      contactPoints: [contactPoint],
      notifiers: [],
      status: [],
    });

    expect(enhanced[0].provenance).toBe(KnownProvenance.File);
  });

  it('should extract provenance from first receiver config that has it', () => {
    const contactPoint: GrafanaManagedContactPoint = {
      name: 'test-contact-point',
      grafana_managed_receiver_configs: [
        {
          uid: 'test-uid-1',
          name: 'test-contact-point',
          type: 'email',
          settings: { addresses: 'test@example.com' },
          secureFields: {},
          // No provenance on first receiver
        },
        {
          uid: 'test-uid-2',
          name: 'test-contact-point',
          type: 'slack',
          settings: { recipient: '#channel' },
          secureFields: {},
          provenance: KnownProvenance.ConvertedPrometheus, // Provenance on second receiver
        },
      ],
    };

    const enhanced = enhanceContactPointsWithMetadata({
      contactPoints: [contactPoint],
      notifiers: [],
      status: [],
    });

    expect(enhanced[0].provenance).toBe(KnownProvenance.ConvertedPrometheus);
  });

  it('should have undefined provenance when neither contact point nor receiver configs have provenance', () => {
    const contactPoint: GrafanaManagedContactPoint = {
      name: 'test-contact-point',
      grafana_managed_receiver_configs: [
        {
          uid: 'test-uid',
          name: 'test-contact-point',
          type: 'email',
          settings: { addresses: 'test@example.com' },
          secureFields: {},
          // No provenance
        },
      ],
    };

    const enhanced = enhanceContactPointsWithMetadata({
      contactPoints: [contactPoint],
      notifiers: [],
      status: [],
    });

    expect(enhanced[0].provenance).toBeUndefined();
  });
});
