---
stage: Create
group: Source Code
info: To determine the technical writer assigned to the Stage/Group associated with this page, see https://handbook.gitlab.com/handbook/product/ux/technical-writing/#assignments
description: GitLabリポジトリ内のコミットとタグに、X.509証明書で署名します。
title: X.509証明書でコミットとタグに署名する
---

{{< details >}}

- プラン: Free、Premium、Ultimate
- 提供形態: GitLab Self-Managed、GitLab Dedicated

{{< /details >}}

[X.509](https://en.wikipedia.org/wiki/X.509)は、公開または非公開の公開キーインフラストラクチャ (PKI) によって発行される公開キー証明書の標準形式です。個人のX.509証明書は、S/MIME（Secure/Multipurpose Internet Mail Extensions）などの認証または署名目的で使用されます。ただし、Gitは[GPG（GnuPGまたはGNU Privacy Guard）](gpg.md)と同様の方法で、X.509証明書を使用したコミットとタグの署名もサポートしています。主な違いは、GitLabがデベロッパーの署名が信頼できるかどうかを判断する方法です:

- X.509の場合、ルート認証局がGitLabトラストストアに追加されます。（トラストストアは、信頼できるセキュリティ証明書のリポジトリです）。署名に必要な中間証明書と組み合わせることで、デベロッパーの証明書を信頼できるルート証明書にチェーンバックできます。
- GPGの場合、デベロッパーは自分の[GPGキー](gpg.md#add-a-gpg-key-to-your-account)をアカウントに追加します。

GitLabは独自の証明書ストアを使用しているため、[トラストチェーン](https://www.ssl.com/article/what-is-a-certificate-authority-ca/)を定義します。コミットまたはタグがGitLabによって検証済みとなるためには、次の条件を満たす必要があります:

- 署名証明書のメールが、GitLabで検証済みのメールアドレスと一致する必要があります。
- GitLabインスタンスには、署名内の証明書からGitLab証明書ストア内の信頼できる証明書への完全なトラストチェーンが必要です。このチェーンには、署名で提供される中間証明書が含まれる場合があります。認証局ルート証明書などの証明書を[GitLab証明書ストアに追加](https://docs.gitlab.com/omnibus/settings/ssl/#install-custom-public-certificates)する必要がある場合があります。
- 署名時刻は、[証明書の有効期間](https://www.rfc-editor.org/rfc/rfc5280.html#section-4.1.2.5)内である必要があります。通常、最大3年です。
- 署名時刻は、コミット時刻と同時またはそれ以降である必要があります。

{{< alert type="note" >}}

トラストチェーン内のルート認証局または中間証明書が期限切れになり、更新された場合、コミットは、[再検証済みにする](#re-verify-commits)まで、一時的に「未検証」と表示されることがあります。

{{< /alert >}}

コミットのステータスがすでに決定され、データベースに保存されている場合は、[ステータスを再確認する](../../../../administration/raketasks/x509_signatures.md)ためにRakeタスクを使用します。[トラブルシューティングセクション](#troubleshooting)を参照してください。GitLabは、バックグラウンドワーカーを使用して、証明書失効リストを毎日確認します。

## 既知の問題 {#known-issues}

- `authorityKeyIdentifier`、`subjectKeyIdentifier`、および`crlDistributionPoints`がない証明書は、**未検証**と表示されます。[RFC 5280](https://www.rfc-editor.org/rfc/rfc5280)に準拠したPKIからの証明書を使用することをお勧めします。
- **検証済み**バッジは、[カスタム認証局](https://docs.gitlab.com/omnibus/settings/ssl/#install-custom-public-certificates)のアップロードがGitLab Self-Managedでのみ可能なため、GitLab SaaSオファリングには表示されません。
- 証明書のExtended Key Usage（EKU）セクションに、必須のKey Usage（KU）`Digital Signature`に加えて値を設定すると、コミットが**未検証**と表示される可能性があります。これを解決するには、EKUリストに`emailProtection`を追加します。RFC 5280に、この制限事項が規定されています。

  診断するには、[OpenSSLを使用したS/MIME検証済み](#smime-verification-with-openssl)に従ってください。この変更でイシューが解決しない場合は、[イシュー440189](https://gitlab.com/gitlab-org/gitlab/-/issues/440189)でフィードバックをお寄せください。
- GitLab 16.2以前のバージョンでは、署名証明書のSubject Alternative Nameリストに複数のメールがある場合、[最初のメールのみがコミットの検証済みに使用されます](https://gitlab.com/gitlab-org/gitlab/-/issues/336677)。

## 署名されたコミットの設定 {#configure-for-signed-commits}

コミット、タグ、またはその両方に署名するには、次の手順を実行する必要があります:

1. [X.509キーペアを取得](#obtain-an-x509-key-pair)。
1. [X.509証明書をGitに関連付け](#associate-your-x509-certificate-with-git)。
1. [コミットに署名して検証済みにする](#sign-and-verify-commits)。
1. [タグに署名して検証済みにする](#sign-and-verify-tags)。

### X.509キーペアの取得 {#obtain-an-x509-key-pair}

組織に公開キーインフラストラクチャ（PKI）がある場合、そのPKIはS/MIMEキーを提供します。PKIからのS/MIMEキーペアがない場合は、独自の自己署名キーペアを作成するか、キーペアを購入してください。

### X.509証明書をGitに関連付け {#associate-your-x509-certificate-with-git}

X.509署名を利用するには、Git 2.19.0以降が必要です。Gitのバージョンは、`git --version`コマンドで確認できます。

正しいバージョンがある場合は、Gitの設定に進むことができます。

{{< tabs >}}

{{< tab title="Linux" >}}

署名にキーを使用するようにGitを設定します:

```shell
signingkey=$( gpgsm --list-secret-keys | egrep '(key usage|ID)' | grep -B 1 digitalSignature | awk '/ID/ {print $2}' )
git config --global user.signingkey $signingkey
git config --global gpg.format x509
```

{{</ tab >}}

{{< tab title="WindowsおよびmacOS" >}}

WindowsまたはmacOSを設定するには:

1. 次のいずれかの方法で[S/MIME Sign](https://github.com/github/smimesign)をインストーラーします:
   - インストーラーをダウンロードします。
   - macOSで`brew install smimesign`を実行します。
1. `smimesign --list-keys`を実行して、証明書のIDを取得します。
1. `git config --global user.signingkey <ID>`を実行して署名キーを設定し、`<ID>`を証明書IDに置き換えます。
1. このコマンドでX.509を設定します:

   ```shell
   git config --global gpg.x509.program smimesign
   git config --global gpg.format x509
   ```

{{< /tab >}}

{{< /tabs >}}

### コミットに署名して検証済みにする {#sign-and-verify-commits}

[X.509証明書をGitに関連付けた](#associate-your-x509-certificate-with-git)後、コミットに署名できます:

1. Gitコミットを作成するときは、`-S`フラグを追加します:

   ```shell
   git commit -S -m "feat: x509 signed commits"
   ```

1. GitLabにプッシュし、`--show-signature`フラグでコミットが検証されていることを確認します:

   ```shell
   git log --show-signature
   ```

1. *コミットするたびに`-S`フラグを入力したくない場合は*、Gitが毎回コミットに署名するように、このコマンドを実行します:

   ```shell
   git config --global commit.gpgsign true
   ```

### タグに署名して検証済みにする {#sign-and-verify-tags}

[X.509証明書をGitに関連付けた](#associate-your-x509-certificate-with-git)後、タグの署名を開始できます:

1. Gitタグを作成するときは、`-s`フラグを追加します:

   ```shell
   git tag -s v1.1.1 -m "My signed tag"
   ```

1. GitLabにプッシュし、このコマンドでタグが署名されていることを確認します:

   ```shell
   git tag --verify v1.1.1
   ```

1. *タグを付けるたびに`-s`フラグを入力したくない場合は*、Gitが毎回タグに署名するように、このコマンドを実行します:

   ```shell
   git config --global tag.gpgsign true
   ```

## 関連トピック {#related-topics}

- [X.509署名のRakeタスク](../../../../administration/raketasks/x509_signatures.md)

## トラブルシューティング {#troubleshooting}

管理者アクセス権を持たないコミッターの場合は、可能な修正について、[署名されたコミットの検証済みの問題](_index.md#fix-verification-problems-with-signed-commits)のリストを確認してください。このページの他のトラブルシューティングの提案には、管理者アクセス権が必要です。

### コミットを再検証する {#re-verify-commits}

GitLabは、チェックされたコミットのステータスをデータベースに保存します。次の後、コミットを再検証済みできます:

- ルート認証局または中間証明書を更新します。
- 証明書ストアに変更を加えます。

コミットを再検証済みするには:

1. ルート認証局とすべての中間証明書がGitLab証明書ストアにあることを確認します。
1. 以前に検証済みのコミットのステータスを確認および更新するには、[`update_signatures`Rakeタスク](../../../../administration/raketasks/x509_signatures.md)を実行します。

### 主な検証済みチェック {#main-verification-checks}

コードは[これらのキーチェック](https://gitlab.com/gitlab-org/gitlab/-/blob/v14.1.0-ee/lib/gitlab/x509/signature.rb#L33)を実行します。これらはすべて`verified`を返す必要があります:

- `x509_certificate.nil?`はfalseである必要があります。
- `x509_certificate.revoked?`はfalseである必要があります。
- `verified_signature`はtrueである必要があります。
- `user.nil?`はfalseである必要があります。
- `user.verified_emails.include?(@email)`はtrueである必要があります。
- `certificate_email == @email`はtrueである必要があります。

コミットが`Unverified`未検証と表示される理由を調査するには:

1. [Railsコンソール](../../../../administration/operations/rails_console.md#starting-a-rails-console-session)を起動します:

   ```shell
   sudo gitlab-rails console
   ```

1. 調査しているプロジェクト（パスまたはID）と完全なコミットSHAを特定します。この情報を使用して`signature`を作成し、他のチェックを実行します:

   ```ruby
   project = Project.find_by_full_path('group/subgroup/project')
   project = Project.find_by_id('121')
   commit = project.repository.commit_by(oid: '87fdbd0f9382781442053b0b76da729344e37653')
   signedcommit=Gitlab::X509::Commit.new(commit)
   signature=Gitlab::X509::Signature.new(signedcommit.signature_text, signedcommit.signed_text, commit.committer_email, commit.created_at, commit.project)
   ```

   チェックを実行して特定されたイシューに対処するために変更を加えた場合は、Railsコンソールを再起動し、最初からもう一度チェックを実行します。

1. コミットの証明書を確認します:

   ```ruby
   signature.x509_certificate.nil?
   signature.x509_certificate.revoked?
   ```

   両方のチェックで`false`が返される必要があります:

   ```ruby
   > signature.x509_certificate.nil?
   => false
   > signature.x509_certificate.revoked?
   => false
   ```

   [既知のイシュー](https://gitlab.com/gitlab-org/gitlab/-/issues/332503)により、これらのチェックは`Validation failed: Subject key identifier is invalid`で失敗します。

1. 署名の暗号学的チェックを実行します。コードは`true`を返す必要があります:

   ```ruby
   signature.verified_signature
   ```

   `false`が返された場合は、[このチェックをさらに調査してください](#cryptographic-verification-checks)。

1. コミットと署名でメールアドレスが一致することを確認します:

   - Railsコンソールには、比較されているメールアドレスが表示されます。
   - 最後のコマンドは`true`を返す必要があります:

   ```ruby
   sigemail=signature.__send__:certificate_email
   commitemail=commit.committer_email
   sigemail == commitemail
   ```

   GitLab 16.2以前では、`Subject Alternative Name`リストの[最初のメールのみ](https://gitlab.com/gitlab-org/gitlab/-/issues/336677)が比較されます。`Subject Alternative Name`リストを表示するには、次を実行します:

   ```ruby
   signature.__send__ :get_certificate_extension,'subjectAltName'
   ```

   デベロッパーのメールアドレスがリストの最初でない場合、このチェックは失敗し、コミットは`unverified`とマークされます。

1. コミットのメールアドレスは、GitLabのアカウントに関連付けられている必要があります。このチェックでは`false`が返されるはずです:

   ```ruby
   signature.user.nil?
   ```

1. メールアドレスがGitLabのユーザーに関連付けられていることを確認します。このチェックでは、`#<User id:1234 @user_handle>`などのユーザーが返されるはずです:

   ```ruby
   User.find_by_any_email(commit.committer_email)
   ```

   `nil`が返された場合、メールアドレスはユーザーに関連付けられておらず、チェックは失敗します。

1. デベロッパーのメールアドレスが検証済みであることを確認します。このチェックでは、trueが返されるはずです:

   ```ruby
   signature.user.verified_emails.include?(commit.committer_email)
   ```

   前のチェックで`nil`が返された場合、このコマンドはエラーを表示します:

   ```plaintext
   NoMethodError (undefined method `verified_emails' for nil:NilClass)
   ```

1. 検証済みステータスはデータベースに保存されます。データベースレコードを表示するには:

   ```ruby
   pp CommitSignatures::X509CommitSignature.by_commit_sha(commit.sha);nil
   ```

   前のチェックがすべて正しい値を返した場合:

   - `verification_status: "unverified"`は、データベースレコードを更新する必要があることを示します。[Rakeタスク](#re-verify-commits)を使用してください。

   - `[]`は、データベースにまだレコードがないことを示します。GitLabでコミットを見つけて署名を確認し、結果を保存します。

#### 暗号学的検証済みチェック {#cryptographic-verification-checks}

GitLabが`verified_signature`が`false`であると判断した場合、Railsコンソールで理由を調査します。これらのチェックでは`signature`が存在する必要があります。前の[主な検証済みチェック](#main-verification-checks)の`signature`ステップを参照してください。

1. 発行者を確認せずに署名を確認すると、`true`が返されます:

   ```ruby
   signature.__send__ :valid_signature?
   ```

1. 署名時刻と日付を確認します。このチェックでは`true`が返される必要があります:

   ```ruby
   signature.__send__ :valid_signing_time?
   ```

   - コードは、コード署名証明書の有効期限が切れることを許可します。
   - コミットは、証明書の有効期間中、およびコミットのデータスタンプ以降に署名する必要があります。`not_before`、`not_after`などのコミット時刻と証明書の詳細を以下で表示します:

     ```ruby
     commit.created_at
     pp signature.__send__ :cert; nil
     ```

1. TLSトラストを確立できることを含め、署名を確認します。このチェックでは`true`が返される必要があります:

   ```ruby
   signature.__send__(:p7).verify([], signature.__send__(:cert_store), signature.__send__(:signed_text))
   ```

   1. これが失敗した場合は、トラストを確立するために必要な不足している証明書を[GitLab証明書ストアに追加](https://docs.gitlab.com/omnibus/settings/ssl/#install-custom-public-certificates)します。

   1. 証明書を追加した後、（これらのトラブルシューティング手順が合格した場合）Rakeタスクを実行して[コミットを再検証済みします](#re-verify-commits)。

   1. 問題を解決するかどうかを確認するために、Railsコンソールで追加の証明書を動的に追加できます。

      1. 変更可能なトラストストア`cert_store`を使用して、署名を再テストします。それでも`false`で失敗するはずです:

         ```ruby
         cert_store = signature.__send__ :cert_store
         signature.__send__(:p7).verify([], cert_store, signature.__send__(:signed_text))
         ```

      1. 追加の証明書を追加して、再テストします:

         ```ruby
         cert_store.add_file("/etc/ssl/certs/my_new_root_ca.pem")
         signature.__send__(:p7).verify([], cert_store, signature.__send__(:signed_text))
         ```

   1. 署名に含まれる証明書を表示します:

      ```ruby
      pp signature.__send__(:p7).certificates ; nil
      ```

   1. [OpenSSLを使用してコマンドラインでさらに調査を実行できます](#smime-verification-with-openssl)。

追加の中間証明書とルート証明書が証明書ストアに追加されていることを確認します。Webサーバーで証明書チェーンが構築される方法との一貫性のため:

- コミットに署名するGitクライアントは、証明書とすべての中間証明書を署名に含める必要があります。
- GitLab証明書ストアには、ルートのみを含める必要があります。

期限切れなどの理由で、GitLabトラストストアからルート証明書を削除すると、そのルートにチェーンバックするコミット署名が`unverified`未検証と表示されます。

#### OpenSSLを使用したS/MIME検証済み {#smime-verification-with-openssl}

署名にイシューがある場合、またはTLSトラストが失敗した場合、コマンドラインでOpenSSLを使用してさらにデバッグを実行できます。

[Railsコンソール](../../../../administration/operations/rails_console.md#starting-a-rails-console-session)から署名と署名付きテキストをエクスポートします:

1. [主な検証済みチェック](#main-verification-checks)の最初の2つのステップは、`signature`が設定されている必要があるため、必須です。

1. OpenSSLでは、PKCS7 PEM形式のデータが`BEGIN PKCS7`と`END PKCS7`で区切られている必要があるため、通常、これを修正する必要があります:

   ```ruby
   pkcs7_text = signature.signature_text.sub('-----BEGIN SIGNED MESSAGE-----', '-----BEGIN PKCS7-----')
   pkcs7_text = pkcs7_text.sub('-----END SIGNED MESSAGE-----', '-----END PKCS7-----')
   ```

1. 署名と署名付きテキストを書き出します:

   ```ruby
   f1=File.new('/tmp/signature_text.pk7.pem','w')
   f1 << pkcs7_text
   f1.close

   f2=File.new('/tmp/signed_text.txt','w')
   f2 << signature.signed_text
   f2.close
   ```

これで、LinuxコマンドラインでOpenSSLを使用してこのデータを調査できます:

1. 署名を含むPKCS＃7ファイルをクエリできます:

   ```shell
   /opt/gitlab/embedded/bin/openssl pkcs7 -inform pem -print_certs \
       -in /tmp/signature_text.pk7.pem -print -noout
   ```

   出力には、少なくとも1つの`cert`セクション（署名者の証明書）が含まれている必要があります。

   出力には、低レベルの詳細がたくさんあります。ここに、存在するはずの構造と見出しの例をいくつか示します:

   ```plaintext
   PKCS7:
     d.sign:
       cert:
           cert_info:
             issuer:
             validity:
               notBefore:
               notAfter:
             subject:
   ```

   デベロッパーのコード署名証明書が中間認証局によって発行されている場合は、追加の証明書の詳細が必要です:

   ```plaintext
   PKCS7:
     d.sign:
       cert:
           cert_info:
       cert:
           cert_info:
   ```

1. 署名から証明書を抽出します:

   ```shell
   /opt/gitlab/embedded/bin/openssl pkcs7 -inform pem -print_certs \
       -in /tmp/signature_text.pk7.pem -out /tmp/signature_cert.pem
   ```

   この手順が失敗した場合、署名に署名者の証明書がない可能性があります。

   - このイシューをGitクライアントで修正します。
   - 次の手順は失敗しますが、署名者の証明書をGitLabサーバーにコピーした場合は、それを使用して`-nointern -certfile signerscertificate.pem`を使用してテストを実行できます。

1. 抽出された証明書を使用して、コミットを部分的に検証済みにします:

   ```shell
   /opt/gitlab/embedded/bin/openssl smime -verify -binary -inform pem \
       -in /tmp/signature_text.pk7.pem -content /tmp/signed_text.txt \
       -noverify -certfile /tmp/signature_cert.pem -nointern
   ```

   出力には通常、次のものが含まれます:

   - 親コミット
   - コミットからの名前、メール、タイムスタンプ
   - コミットテキスト
   - `Verification successful`（または類似）

   このチェックは、GitLabが実行するチェックと同じではありません。なぜなら:

   - 署名者の証明書を検証済みにしない（`-noverify`）
   - 検証済みは、メッセージ内のもの（`-nointern`）ではなく、指定された`-certfile`を使用して行われます

1. メッセージ内の証明書を使用して、コミットを部分的に検証済みにします:

   ```shell
   /opt/gitlab/embedded/bin/openssl smime -verify -binary -inform pem \
       -in /tmp/signature_text.pk7.pem -content /tmp/signed_text.txt \
       -noverify
   ```

   抽出された証明書を使用して、前のステップと同じ結果が得られるはずです。

   メッセージに証明書がない場合、エラーには`signer certificate not found`が含まれます。

1. コミットを完全に検証済みにします:

   ```shell
   /opt/gitlab/embedded/bin/openssl smime -verify -binary -inform pem \
       -in /tmp/signature_text.pk7.pem -content /tmp/signed_text.txt
   ```

   このステップが失敗した場合、GitLabでも検証済みが失敗します。

   たとえば、エラーを解決します:

   - `certificate verify error .. unable to get local issuer certificate`: 
     - トラストチェーンを確立できませんでした。
     - このOpenSSLバイナリは、GitLabトラストストアを使用します。ルート証明書がトラストストアにないか、署名に中間証明書がなく、信頼できるルートへのチェーンを構築できません。
       - 署名に中間証明書を含めることができない場合は、中間証明書をトラストストアに配置できます。
       - [証明書を追加する手順](https://docs.gitlab.com/omnibus/settings/ssl/#install-custom-public-certificates) \- パッケージ化されたGitLabのトラストストア用 - `/etc/gitlab/trusted-certs`を使用。
     - OpenSSLを使用して追加の信頼できる証明書をテストします: `-CAfile /path/to/rootcertificate.pem`
   - `unsupported certificate purpose`: 
     - 証明書は、署名者の証明書の`X509v3 Key Usage`セクションで`Digital Signature`を指定する必要があります。
     - `X509v3 Extended Key Usage`（EKU）セクションが指定されている場合は、`emailProtection`を含める必要があります。詳細については、[RFC 5280](https://datatracker.ietf.org/doc/html/rfc5280#page-44)を参照してください:

       > （Key Usage）拡張の両方と矛盾しない目的がない場合、証明書はいかなる目的にも使用できません。

       EKUリストへのこの追加でイシューが解決されない場合は、[issue 440189](https://gitlab.com/gitlab-org/gitlab/-/issues/440189)でフィードバックをお寄せください。
   - `signer certificate not found`、次のいずれかです:
     - 引数`-nointern`を追加しましたが、`-certfile`を提供していません。
     - 署名に署名者の証明書がありません。
