# Copyright (C) 2024 Intel Corporation
# SPDX-License-Identifier: MIT

import os
import shutil
import sys
from dataclasses import dataclass
from typing import List, Union

from .command import Command
from .. import project_root_path
from ..installer.installers import BasicProductionInstaller, FullProductionInstaller


@dataclass
class VENV_args:
    path_str: str = '.venv'
    clean: bool  = False
    verbose: bool = False


class VirtualEnvironmentVersion:
    MIN_EXPECTED_VERSION = 15

    def __init__(self):
        self.__path_to_python = sys.executable
        self.__major = 0
        self.__minor = 0
        self.__revision= 0

    def validate(self):
        if VirtualEnvironment.is_active():
            return
        self.__get_version()
        if int(self.__major) < self.MIN_EXPECTED_VERSION:
            raise Exception(self.__get_exception_message())

    def __get_version(self):
        try:
            (self.__major, self.__minor, self.__revision) = tuple(Command.run(
                [self.__path_to_python, '-c', 'import virtualenv; print(virtualenv.__version__)'],
                never_echo=True).strip().split('.'))
        except Exception as e:
            print(f'Failed to get virtualenv version: {e}')

    def __get_exception_message(self):
        return \
            f'virtualenv version {self.MIN_EXPECTED_VERSION} or greater required\n' \
            f'Current version: {self.__major}.{self.__minor}.{self.__revision}\n' \
            f'Install virtualenv:\n' \
            f'  pip install virtualenv\n' \
            f'Or upgrade virtualenv and pip:\n' \
            f'  pip install --upgrade pip virtualenv\n'


class VirtualEnvironment:
    VIRTUAL_ENV = 'VIRTUAL_ENV'
    DEFAULT_VENV = VENV_args().path_str
    DEFAULT_VENV_PATH = os.path.join(project_root_path, DEFAULT_VENV)
    VALID_INSTALLERS_UNION = Union[BasicProductionInstaller, FullProductionInstaller, None]

    def __init__(self, venv_args: VENV_args, installer: VALID_INSTALLERS_UNION = None):
        VirtualEnvironmentVersion().validate()
        self.__venv_args = venv_args
        self.__installer = installer if installer else BasicProductionInstaller(self.__venv_args.verbose)
        self.__installer.update_python_path(self.__get_python_path())

    @property
    def venv_path(self) -> str:
        return os.path.abspath(self.__venv_args.path_str)

    @property
    def installer(self) -> Union[BasicProductionInstaller]:
        return self.__installer

    def setup(self):
        self.clean()
        self.create()
        self.activate()
        self.install()

    def clean(self):
        if self.is_active():
            return
        if self.exists():
            if self.__venv_args.clean:
                self.__clean()
            else:
                print(f'Virtual environment {self.venv_path} already exists.\n'
                      f'To recreate a clean version of this virtual environment, use --venv-clean.\n'
                      f'To specify a different virtual environment path use --venv-path VENV_PATH.\n'
                      f'To learn more about installer options, use --help.\n'
                      f'To update an existing virtual environment, activate the virtual environment and '
                      f'run the installer again without venv options within the active environment.\n'
                      )
                exit(0)

    def create(self):
        if not os.path.isdir(self.venv_path):
            self.__create()

    def activate(self):
        if self.is_active():
            print(f'Using current virtual environment: {self.get_active_virtualenv_path()}')
        else:
            print(f'Activating virtual environment {self.venv_path}...')
            activate_script = self.__get_activate_script()
            exec(compile(open(activate_script, "rb").read(), activate_script, 'exec'), dict(__file__=activate_script))

    def install(self):
        self.__installer.install()

    @staticmethod
    def is_active() -> bool:
        return VirtualEnvironment.VIRTUAL_ENV in os.environ

    def exists(self):
        return os.path.isdir(self.venv_path)

    @staticmethod
    def get_active_virtualenv_path() -> str:
        if VirtualEnvironment.is_active():
            return os.environ[VirtualEnvironment.VIRTUAL_ENV]
        raise RuntimeError('No active virtual environment found')

    @staticmethod
    def get_active_virtualenv_python_path() -> str:
        venv_path = VirtualEnvironment.get_active_virtualenv_path()
        script_path = VirtualEnvironment.__get_venv_script_path()
        return os.path.join(venv_path, script_path, 'python')

    @staticmethod
    def get_venv_activate_command(venv_path: str) -> str:
        script_path = VirtualEnvironment.__get_venv_script_path()
        activate_script = os.path.join(venv_path, script_path, 'activate')
        if sys.platform == 'win32':
            return f'{activate_script}'
        return f'source {activate_script}'

    def __get_activate_script(self):
        return os.path.join(self.venv_path, self.__get_venv_script_path(), 'activate_this.py')

    def __get_python_path(self):
        return os.path.join(self.venv_path, self.__get_venv_script_path(), 'python')

    def __create(self):
        print(f'Creating virtual environment {self.venv_path}...')
        self.__run_virtualenv_cmd([self.venv_path])

    def __clean(self):
        if os.path.isdir(self.venv_path):
            try:
                print(f'Removing virtualenv {self.venv_path}...')
                shutil.rmtree(self.venv_path)
            except PermissionError as e:
                print(f'The installer doesn\'t have permission to remove virtualenv {self.venv_path}')
                print('Please manually remove it and try again or use \'--venv-path VENV_PATH\' '
                         'to specify a new virtual environment directory.')
                exit(0)

    @staticmethod
    def __get_venv_script_path():
        WINDOWS_SCRIPT_PATH = 'Scripts'
        LINUX_SCRIPT_PATH = 'bin'
        return WINDOWS_SCRIPT_PATH if sys.platform == 'win32' else LINUX_SCRIPT_PATH

    @staticmethod
    def __run_virtualenv_cmd(cmd_args: List[str]):
        cmd = [sys.executable, '-m', 'virtualenv'] + cmd_args
        try:
            Command.run(cmd, never_echo=True)
        except Exception as e:
            print(f'Failed to run command: {cmd}')
            raise e


