/*******************************************************************************
* Copyright (C) 2014 Intel Corporation
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

//@HEADER
// ***************************************************
//
// HPCG: High Performance Conjugate Gradient Benchmark
//
// Contact:
// Michael A. Heroux ( maherou@sandia.gov)
// Jack Dongarra     (dongarra@eecs.utk.edu)
// Piotr Luszczek    (luszczek@eecs.utk.edu)
//
// ***************************************************
//@HEADER

/*!
 @file ComputeMG.cpp

 HPCG routine
 */

#include "ComputeMG.hpp"
#include "ComputeSYMGS.hpp"
#include "ComputeSPMV.hpp"
#include "ComputeRestriction_ref.hpp"
#include "ComputeProlongation_ref.hpp"

#ifndef HPCG_NO_MPI
#include "ExchangeHalo.hpp"
#include <mpi.h>
#include "Geometry.hpp"
#include <cstdlib>
#endif

#include "VeryBasicProfiler.hpp"

#ifdef BASIC_PROFILING
#define BEGIN_PROFILE(n) optData->profiler->begin(n);
#define END_PROFILE(n) optData->profiler->end(n);
#define END_PROFILE_WAIT(n, event) event.wait(); optData->profiler->end(n);
#else
#define BEGIN_PROFILE(n)
#define END_PROFILE(n)
#define END_PROFILE_WAIT(n, event)
#endif

int ComputeMG_ref(const SparseMatrix & A, const Vector & r, Vector & x);

/*!
  @param[in] A the known system matrix
  @param[in] r the input vector
  @param[inout] x On exit contains the result of the multigrid V-cycle with r as the RHS, x is the approximation to Ax = r.

  @return returns 0 upon success and non-zero otherwise

  @see ComputeMG_ref
*/
sycl::event ComputeMG(const SparseMatrix  & A, const Vector & r, Vector & x, sycl::queue & main_queue,
                      int& ierr, const std::vector<sycl::event> & deps)
{
#ifdef HPCG_LOCAL_LONG_LONG
    ComputeMG_ref(A,r,x);
#else
#ifdef BASIC_PROFILING
    struct optData *optData = (struct optData *)A.optimizationData;
#endif
    if (A.mgData!=0) // Go to next coarse level if defined
    {
        const int numberOfPresmootherSteps = A.mgData->numberOfPresmootherSteps;
        const int numberOfPostsmootherSteps = A.mgData->numberOfPostsmootherSteps;
        sycl::event smoothing_ev;
        if ( numberOfPresmootherSteps > 1 )
        {
            // this code path is not take for a standard HPCG run (pre/post smoothing steps == 1)
            smoothing_ev = ComputeSYMGS(A, r, x, main_queue, ierr, deps);
            for ( int i = 2; i < numberOfPresmootherSteps; ++i ) {
                smoothing_ev = ComputeSYMGS(A, r, x, main_queue, ierr, {smoothing_ev});
            }
            smoothing_ev = ComputeSPMV(A, x, (*A.mgData->Axf), main_queue, ierr, {smoothing_ev});
        } else
        {
            smoothing_ev = ComputeSYMGS_MV(A, r, x, (*A.mgData->Axf), main_queue, ierr, deps);
        }

        BEGIN_PROFILE("coarselevels");
        auto restrict_ev = ComputeRestriction(A, r, main_queue, ierr, {smoothing_ev});
        auto mg_ev = ComputeMG(*A.Ac,*A.mgData->rc, *A.mgData->xc, main_queue, ierr, {restrict_ev});
        auto chain_ev = ComputeProlongation(A, x, main_queue, ierr, {mg_ev});
        END_PROFILE_WAIT("coarselevels", chain_ev);
        for ( int i = 0; i < numberOfPostsmootherSteps; ++i ) {
            chain_ev = ComputeSYMGS(A, r, x, main_queue, ierr, {chain_ev});
        }
        return chain_ev;

    } else
    {
        auto zv_ev = ZeroVector(x, main_queue, deps);
        return ComputeSYMGS(A, r, x, main_queue, ierr, {zv_ev});
    }
#endif
    return sycl::event();
}
