#
# Copyright (C) 2017 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of them
# is governed by the express license under which they were provided to you ("License"). Unless
# the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
# or transmit this software or the related documents without Intel's prior written permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#


# ------------------------------------------------------------------------------
# This example shows how to obtain the cache simulation data.
# ------------------------------------------------------------------------------

import sys

try:

    import advisor

except ImportError:

    print(
        """Import Error: Python could not resolve path to Advisor's pythonapi directory.
        To fix, either manually add path to the pythonapi directory into PYTHONPATH environment
        variable, or use advixe-vars.* scripts to set up product environment variables automatically."""
    )
    sys.exit(1)

# Check command-line arguments.
if len(sys.argv) < 2:
    print('Usage: "python {} path_to_project_dir"'.format(__file__))
    sys.exit(2)

# Open the Advisor Project and load the data.
project = advisor.open_project(sys.argv[1])

try:

    data = project.load(advisor.MAP)

except OSError as _:

    print("Advisor profile does not contain MAP data.")
    sys.exit()

# Iteratve over each loop and print the cache simulation data.
for site in data.map:
    indent = "    "
    width = 19
    site_id = site["site_id"]
    cachesim = data.get_cachesim_info(site_id)
    if not cachesim:
        break

    attrs = cachesim.attrs
    if not attrs:
        continue

    print("Site: " + site["site_name"])

    print(indent + "Location: " + site["site_name"])
    print(indent * 2 + "File: " + site["site_source_file"])
    print(indent * 2 + "Line: " + site["site_source_line"])

    print("")
    print(indent + "Cache model settings:")
    print(indent * 2 + "Associativity".ljust(width) + " = {}".format(attrs["associativity"]))
    print(indent * 2 + "Sets".ljust(width) + " = {}".format(attrs["sets"]))
    print("")

    print(indent + "Cache model results:")
    if "writes" in attrs:
        print(indent * 2 + "Writes".ljust(width) + " = {}".format(attrs["writes"]))
    if "reads" in attrs:
        print(indent * 2 + "Reads".ljust(width) + " = {}".format(attrs["reads"]))
    if "read_misses" in attrs:
        print(indent * 2 + "Read misses".ljust(width) + " = {}".format(attrs["read_misses"]))
    print("")

    print(indent + "Evicted cache lines utilization:")
    print(indent * 2 + "Average utilization".ljust(width) + " = {:.2f}%".format(cachesim.utilization))
    print("")
    rows = [[str(v) for v in row] for row in cachesim.evicted]
    widths = [0, 0]
    for col in range(2):
        for row in rows:
            widths[col] = max(widths[col], len(row[col]))
    for row in rows:
        print(indent * 2 + row[0].rjust(widths[0]) + " | " + row[1].rjust(widths[1]))
