# Copyright (C) 2024 Intel Corporation
# SPDX-License-Identifier: MIT

import sys
import logging

from mpp.versioning.mpp_version import version_info
from mpp.core.types import VerboseLevel
from mpp.console_output import ConsoleOutput


class CliConsoleOutput(ConsoleOutput):
    """
    A set of static methods to output information to the user console in MPP CLI
    """

    @staticmethod
    def initialize_logging(verbose):
        logging_level = CliConsoleOutput._get_logging_level(verbose)
        logging.basicConfig(level=logging_level, format='%(asctime)s %(levelname)s: %(message)s',
                    datefmt='%Y-%m-%d %I:%M:%S %p', stream=sys.stderr)

    @staticmethod
    def show_run_information(ref_tsc):
        print('System information:')
        print(f'     TSC Frequency: {ref_tsc // 1000000} MHz')
        print('\nProcessing data: ...')

    @staticmethod
    def show_banner():
        print(f'Metric post processor version {version_info.get_version_and_build_id()}')
        print(f'Python {sys.version}')

    @staticmethod
    def show_tps_views_generated(number_of_seconds):
        print(f'Generated all Transactions Per Second tables in {number_of_seconds}')

    @staticmethod
    def show_excel_file_creation():
        print('Creating Excel output file...')

    @staticmethod
    def show_excel_output_destination(output_file_specifier):
        print(f'Output written to: {output_file_specifier}')

    @staticmethod
    def show_error(msg, exit_code=1, exception=None):
        """
        Print an error and terminate the script
        :param msg: error message to print
        :param exit_code: optional exit code (default is 1)
        :param exception: optional exception object
        """
        logging.exception(exception)
        if exception:
            msg += f': {exception}'
        print(msg, file=sys.stderr)
        exit(exit_code)

    @staticmethod
    def show_sample_count(first_sample, last_sample, num_unique_events, total_metrics_derived):
        sample_count = last_sample - first_sample + 1
        print(f'{sample_count} samples processed')
        print(f'{num_unique_events} events parsed and {len(total_metrics_derived)} metrics derived.')

    @staticmethod
    def _get_logging_level(verbose):
        if verbose == VerboseLevel.DEBUG:
            logging_level = logging.DEBUG
        else:
            logging_level = logging.WARNING
        # TODO: add info and batch_info logging levels here
        return logging_level
